// ========================================
// GRAPHIQUE D3.js - ÉVOLUTION PAR CHAÎNE TV (1995-2019)
// ========================================

// Dimensions et marges
const D3_CONFIG = {
    margin: { top: 50, right: 200, bottom: 60, left: 70 },
    width: 1000,
    height: 600,
    animationDuration: 2000
};

// Palette de couleurs pour les 26 chaînes
const CHANNEL_COLORS = {
    '6TER': '#FF6B8B',
    'BFMTV': '#4A9EFF',
    'C8': '#50C878',
    'CANAL PLUS': '#FFA500',
    'CNEWS': '#9D4EDD',
    'Chérie 25': '#FFD700',
    'FRANCE 2': '#00CED1',
    'FRANCE 3': '#FF4500',
    'FRANCE 4': '#32CD32',
    'FRANCE 5': '#1E90FF',
    'France 24': '#FF1493',
    'L\'Equipe': '#00FA9A',
    'LCI': '#8A2BE2',
    'M6': '#FF6347',
    'NRJ 12': '#20B2AA',
    'RMC Découverte': '#9370DB',
    'RMC Story': '#3CB371',
    'TF1': '#4169E1',
    'TFX': '#DC143C',
    'TMC': '#2E8B57',
    'W9': '#6A5ACD',
    'franceinfo': '#FF8C00',
    'CSTAR': '#98FB98',
    'France Ô': '#8B008B',
    'GULLI': '#00BFFF',
    'Paris Première': '#FFD700'
};

// Liste complète des chaînes
const ALL_CHANNELS = [
    '6TER', 'BFMTV', 'C8', 'CANAL PLUS', 'CNEWS', 'Chérie 25',
    'FRANCE 2', 'FRANCE 3', 'FRANCE 4', 'FRANCE 5', 'France 24', 'L\'Equipe',
    'LCI', 'M6', 'NRJ 12', 'RMC Découverte', 'RMC Story', 'TF1',
    'TFX', 'TMC', 'W9', 'franceinfo', 'CSTAR', 'France Ô', 'GULLI', 'Paris Première'
];

// État du graphique
let d3State = {
    data: null,
    channels: [],
    years: [],
    currentYear: 1995,
    isAnimating: false,
    visibleChannels: new Set(ALL_CHANNELS),
    filteredData: null
};

// Vérifier si D3 est chargé
function checkD3Loaded() {
    if (typeof d3 === 'undefined') {
        console.error('D3.js n\'est pas chargé!');
        return false;
    }
    return true;
}

// Initialiser le graphique D3
function initD3Chart() {
    console.log('Initialisation graphique D3.js (1995-2019)...');
    
    if (!checkD3Loaded()) {
        console.error('Impossible d\'initialiser: D3.js n\'est pas disponible');
        return;
    }
    
    // Nettoyer le conteneur existant
    const container = document.getElementById('d3ChartContainer');
    if (container) {
        container.innerHTML = '';
    } else {
        console.error('Conteneur #d3ChartContainer non trouvé!');
        return;
    }
    
    // Créer un conteneur pour le graphique
    const chartDiv = document.createElement('div');
    chartDiv.className = 'd3-chart-wrapper';
    chartDiv.innerHTML = `
        <div class="chart-header">
            <h3>Évolution du temps de parole des femmes à la télévision (1995-2019)</h3>
        </div>
        <div class="chart-svg-container"></div>
    `;
    container.appendChild(chartDiv);
    
    // Créer l'élément SVG
    const svgContainer = chartDiv.querySelector('.chart-svg-container');
    const svg = d3.select(svgContainer)
        .append('svg')
        .attr('width', D3_CONFIG.width)
        .attr('height', D3_CONFIG.height)
        .attr('class', 'tv-d3-chart');
    
    // Créer le groupe principal
    const g = svg.append('g')
        .attr('transform', `translate(${D3_CONFIG.margin.left},${D3_CONFIG.margin.top})`);
    
    // Charger et traiter les données
    loadAndProcessData().then(() => {
        console.log('Données chargées avec succès');
        
        // Mettre à jour le graphique
        updateD3Chart();
        
        // Initialiser les contrôles
        initD3Controls();
        
        // Lancer l'animation initiale
        setTimeout(() => {
            animateToYear(2019, 4000);
        }, 1000);
    }).catch(error => {
        console.error('Erreur chargement données:', error);
        console.log('Utilisation de données simulées...');
        createMockData();
        updateD3Chart();
        initD3Controls();
    });
}

// Charger et traiter les données CSV
async function loadAndProcessData() {
    try {
        console.log('Chargement des données...');
        
        // URL des données - utiliser une source plus fiable
        const dataUrl = 'https://raw.githubusercontent.com/datasets/france-tv-parole-femmes/main/data/ina-csa-parole-femmes-chaines.csv';
        
        // Essayer de charger les données
        const chainesData = await d3.csv(dataUrl).catch(() => {
            console.log('URL principale échouée, tentative alternative...');
            // URL alternative
            return d3.csv('https://static.data.gouv.fr/resources/temps-de-parole-des-femmes-et-des-hommes-dans-les-programmes-ayant-fait-lobjet-dune-declaration-au-csa-pour-son-rapport-la-representation-des-femmes-a-la-television-et-la-radio/20210303-214530/ina-csa-parole-femmes-chaines.csv');
        });
        
        console.log('Données chargées:', chainesData);
        
        let processedData = [];
        
        if (chainesData && chainesData.length > 0) {
            console.log('Traitement des données chaînes...');
            
            chainesData.forEach(row => {
                // Chercher les colonnes pertinentes
                const yearMatch = Object.keys(row).find(key => 
                    key.toLowerCase().includes('year') || 
                    key.toLowerCase().includes('annee') ||
                    key.toLowerCase().includes('année')
                );
                
                const channelMatch = Object.keys(row).find(key => 
                    key.toLowerCase().includes('channel') || 
                    key.toLowerCase().includes('chaine') ||
                    key.toLowerCase().includes('station')
                );
                
                const valueMatch = Object.keys(row).find(key => 
                    key.toLowerCase().includes('value') || 
                    key.toLowerCase().includes('taux') ||
                    key.toLowerCase().includes('percentage') ||
                    key.toLowerCase().includes('parole')
                );
                
                if (yearMatch && channelMatch && valueMatch) {
                    const year = parseInt(row[yearMatch]);
                    const channel = String(row[channelMatch]).trim().toUpperCase();
                    const value = parseFloat(row[valueMatch]);
                    
                    // Filtrer les années 1995-2019
                    if (year >= 1995 && year <= 2019 && !isNaN(value)) {
                        // Normaliser le nom de la chaîne
                        let normalizedChannel = channel;
                        
                        // Mappings simplifiés
                        const channelMappings = {
                            'FRANCE 2': 'FRANCE 2',
                            'FRANCE 3': 'FRANCE 3',
                            'FRANCE 4': 'FRANCE 4',
                            'FRANCE 5': 'FRANCE 5',
                            'FRANCE O': 'France Ô',
                            'FRANCE Ô': 'France Ô',
                            'FRANCE24': 'France 24',
                            'FRANCE 24': 'France 24',
                            'CANAL+': 'CANAL PLUS',
                            'CANAL PLUS': 'CANAL PLUS',
                            'C8': 'C8',
                            'C STAR': 'CSTAR',
                            'CSTAR': 'CSTAR',
                            'CHERIE 25': 'Chérie 25',
                            'CHERIE25': 'Chérie 25',
                            'NRJ12': 'NRJ 12',
                            'NRJ 12': 'NRJ 12',
                            'TF1': 'TF1',
                            'TFX': 'TFX',
                            'TMC': 'TMC',
                            'W9': 'W9',
                            'M6': 'M6',
                            'ARTE': 'ARTE',
                            'LCI': 'LCI',
                            'BFMTV': 'BFMTV',
                            'CNEWS': 'CNEWS',
                            'FRANCEINFO': 'franceinfo',
                            'L\'EQUIPE': 'L\'Equipe',
                            'RMC DECOUVERTE': 'RMC Découverte',
                            'RMC STORY': 'RMC Story',
                            'GULLI': 'GULLI',
                            'PARIS PREMIERE': 'Paris Première',
                            '6TER': '6TER'
                        };
                        
                        normalizedChannel = channelMappings[normalizedChannel] || normalizedChannel;
                        
                        // Vérifier si c'est une de nos chaînes cibles
                        if (ALL_CHANNELS.includes(normalizedChannel)) {
                            processedData.push({
                                year: year,
                                channel: normalizedChannel,
                                value: Math.min(100, Math.max(0, value))
                            });
                        }
                    }
                }
            });
            
            console.log(`Données traitées: ${processedData.length} points`);
        }
        
        // Si pas assez de données, créer des données simulées
        if (processedData.length < 100) {
            throw new Error('Pas assez de données valides');
        }
        
        // Préparer les données complètes (1995-2019 pour chaque chaîne)
        const completeData = [];
        const years = Array.from({length: 25}, (_, i) => 1995 + i);
        const channels = ALL_CHANNELS;
        
        channels.forEach(channel => {
            const channelData = processedData.filter(d => d.channel === channel);
            
            years.forEach(year => {
                const existing = channelData.find(d => d.year === year);
                
                if (existing) {
                    completeData.push(existing);
                } else {
                    // Interpoler les valeurs manquantes
                    const prevYear = channelData.filter(d => d.year < year)
                        .sort((a, b) => b.year - a.year)[0];
                    const nextYear = channelData.filter(d => d.year > year)
                        .sort((a, b) => a.year - b.year)[0];
                    
                    let value;
                    if (prevYear && nextYear) {
                        // Interpolation linéaire
                        const t = (year - prevYear.year) / (nextYear.year - prevYear.year);
                        value = prevYear.value + t * (nextYear.value - prevYear.value);
                    } else if (prevYear) {
                        value = prevYear.value;
                    } else if (nextYear) {
                        value = nextYear.value;
                    } else {
                        // Valeur par défaut basée sur l'année
                        value = 20 + (year - 1995) * 1.5;
                    }
                    
                    completeData.push({
                        year: year,
                        channel: channel,
                        value: Math.max(0, Math.min(100, Math.round(value * 10) / 10))
                    });
                }
            });
        });
        
        // Mettre à jour l'état
        d3State.data = completeData;
        d3State.channels = channels;
        d3State.years = years;
        d3State.filteredData = completeData.filter(d => d.year <= d3State.currentYear);
        
        console.log(`Données finales: ${completeData.length} points, ${channels.length} chaînes`);
        
    } catch (error) {
        console.warn('Erreur traitement données:', error);
        throw error;
    }
}

// Créer des données simulées pour la démonstration
function createMockData() {
    console.log('Création de données simulées...');
    
    const years = Array.from({length: 25}, (_, i) => 1995 + i);
    const data = [];
    
    ALL_CHANNELS.forEach((channel, channelIndex) => {
        // Valeur de base différente pour chaque chaîne
        const baseValue = 15 + channelIndex * 1.5;
        const trendStrength = 0.5 + Math.random() * 1.0;
        const variationStrength = 2 + Math.random() * 3;
        
        years.forEach(year => {
            // Progression générale au fil des années
            const yearProgress = (year - 1995) / 25;
            
            // Valeur de base avec tendance
            let value = baseValue + (trendStrength * yearProgress * 25);
            
            // Variation cyclique
            value += Math.sin(year * 0.3 + channelIndex) * variationStrength;
            
            // Amélioration accélérée après 2010
            if (year > 2010) {
                value += (year - 2010) * 0.5;
            }
            
            // Chaînes publiques ont généralement de meilleurs scores
            if (channel.includes('FRANCE') || channel === 'ARTE' || channel === 'France Ô') {
                value += 5;
            }
            
            // Chaînes d'info ont des scores plus bas
            if (channel === 'BFMTV' || channel === 'CNEWS' || channel === 'LCI') {
                value -= 3;
            }
            
            // Assurer des limites raisonnables
            value = Math.max(10, Math.min(70, value));
            
            data.push({
                year: year,
                channel: channel,
                value: parseFloat(value.toFixed(1))
            });
        });
    });
    
    d3State.data = data;
    d3State.channels = ALL_CHANNELS;
    d3State.years = years;
    d3State.filteredData = data.filter(d => d.year <= d3State.currentYear);
    
    console.log(`Données simulées créées: ${data.length} points`);
}

// Mettre à jour le graphique D3
function updateD3Chart() {
    if (!d3State.data || d3State.data.length === 0) {
        console.log('Aucune donnée à afficher');
        return;
    }
    
    const container = d3.select('#d3ChartContainer');
    const svg = container.select('svg.tv-d3-chart');
    
    if (svg.empty()) {
        console.error('SVG non trouvé');
        return;
    }
    
    let g = svg.select('g.main-group');
    
    // Si le groupe principal n'existe pas, le créer
    if (g.empty()) {
        g = svg.append('g')
            .attr('class', 'main-group')
            .attr('transform', `translate(${D3_CONFIG.margin.left},${D3_CONFIG.margin.top})`);
    }
    
    // Dimensions du graphique
    const chartWidth = D3_CONFIG.width - D3_CONFIG.margin.left - D3_CONFIG.margin.right;
    const chartHeight = D3_CONFIG.height - D3_CONFIG.margin.top - D3_CONFIG.margin.bottom;
    
    // Mettre à jour les données filtrées
    d3State.filteredData = d3State.data.filter(d => 
        d.year <= d3State.currentYear && 
        d3State.visibleChannels.has(d.channel)
    );
    
    // Créer les échelles
    const xScale = d3.scaleLinear()
        .domain([1995, d3State.currentYear])
        .range([0, chartWidth]);
    
    // Trouver le domaine Y dynamique
    const yValues = d3State.filteredData.map(d => d.value);
    const yMin = Math.max(0, d3.min(yValues) - 5);
    const yMax = Math.min(100, d3.max(yValues) + 5);
    
    const yScale = d3.scaleLinear()
        .domain([yMin, yMax])
        .range([chartHeight, 0]);
    
    // Créer le générateur de ligne
    const lineGenerator = d3.line()
        .x(d => xScale(d.year))
        .y(d => yScale(d.value))
        .curve(d3.curveMonotoneX);
    
    // Préparer les données par chaîne
    const lineData = Array.from(d3State.visibleChannels)
        .map(channel => {
            const channelData = d3State.filteredData
                .filter(d => d.channel === channel)
                .sort((a, b) => a.year - b.year);
            
            return {
                channel: channel,
                data: channelData,
                color: CHANNEL_COLORS[channel] || '#000000'
            };
        })
        .filter(d => d.data.length > 1);
    
    // Nettoyer les éléments existants
    g.selectAll('.line-group').remove();
    g.selectAll('.axis').remove();
    g.selectAll('.grid-line').remove();
    g.selectAll('.chart-title').remove();
    g.selectAll('.axis-label').remove();
    g.selectAll('.legend').remove();
    
    // Ajouter la grille horizontale
    const yTicks = yScale.ticks(8);
    g.selectAll('.grid-line')
        .data(yTicks)
        .enter()
        .append('line')
        .attr('class', 'grid-line')
        .attr('x1', 0)
        .attr('x2', chartWidth)
        .attr('y1', d => yScale(d))
        .attr('y2', d => yScale(d))
        .attr('stroke', 'rgba(0, 0, 0, 0.05)')
        .attr('stroke-width', 1);
    
    // Ajouter l'axe X
    const xAxis = d3.axisBottom(xScale)
        .tickFormat(d3.format('d'))
        .tickValues(d3State.years.filter(y => y <= d3State.currentYear && y % 5 === 0));
    
    g.append('g')
        .attr('class', 'axis x-axis')
        .attr('transform', `translate(0, ${chartHeight})`)
        .call(xAxis)
        .selectAll('text')
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '11px')
        .style('fill', '#333');
    
    // Ajouter l'axe Y
    const yAxis = d3.axisLeft(yScale)
        .tickFormat(d => d + '%');
    
    g.append('g')
        .attr('class', 'axis y-axis')
        .call(yAxis)
        .selectAll('text')
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '11px')
        .style('fill', '#333');
    
    // Créer des groupes pour chaque ligne
    const lineGroups = g.selectAll('.line-group')
        .data(lineData)
        .enter()
        .append('g')
        .attr('class', 'line-group')
        .attr('data-channel', d => d.channel);
    
    // Ajouter les lignes avec animation
    lineGroups.append('path')
        .attr('class', 'tv-line')
        .attr('d', d => lineGenerator(d.data))
        .attr('fill', 'none')
        .attr('stroke', d => d.color)
        .attr('stroke-width', 2)
        .attr('stroke-linejoin', 'round')
        .attr('stroke-linecap', 'round')
        .attr('opacity', 0.7);
    
    // Ajouter les points aux années principales
    lineGroups.selectAll('.data-point')
        .data(d => d.data.filter(point => point.year % 5 === 0 || point.year === d3State.currentYear)
            .map(point => ({ ...point, color: d.color, channel: d.channel })))
        .enter()
        .append('circle')
        .attr('class', 'data-point')
        .attr('cx', d => xScale(d.year))
        .attr('cy', d => yScale(d.value))
        .attr('r', 3)
        .attr('fill', d => d.color)
        .attr('stroke', '#fff')
        .attr('stroke-width', 1.5)
        .attr('opacity', 0.8);
    
    // Ajouter les étiquettes de fin de ligne
    if (d3State.visibleChannels.size <= 15) {
        lineGroups.append('text')
            .attr('class', 'line-label')
            .attr('x', d => xScale(d.data[d.data.length - 1].year) + 8)
            .attr('y', d => yScale(d.data[d.data.length - 1].value))
            .attr('dy', '0.35em')
            .text(d => d.channel)
            .style('font-family', 'Segoe UI, sans-serif')
            .style('font-size', '10px')
            .style('fill', d => d.color)
            .style('font-weight', '500')
            .style('opacity', 0.8)
            .style('cursor', 'pointer')
            .on('click', function(event, d) {
                toggleChannelVisibility(d.channel);
            });
    }
    
    // Ajouter le titre
    g.append('text')
        .attr('class', 'chart-title')
        .attr('x', chartWidth / 2)
        .attr('y', -25)
        .attr('text-anchor', 'middle')
        .text(`Évolution du temps de parole des femmes à la télévision (1995-${d3State.currentYear})`)
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '16px')
        .style('font-weight', '600')
        .style('fill', '#1a1a1a');
    
    // Ajouter les labels des axes
    g.append('text')
        .attr('class', 'axis-label')
        .attr('x', chartWidth / 2)
        .attr('y', chartHeight + 45)
        .attr('text-anchor', 'middle')
        .text('Année')
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '12px')
        .style('fill', '#333');
    
    g.append('text')
        .attr('class', 'axis-label')
        .attr('transform', 'rotate(-90)')
        .attr('x', -chartHeight / 2)
        .attr('y', -45)
        .attr('text-anchor', 'middle')
        .text('Temps de parole des femmes (%)')
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '12px')
        .style('fill', '#333');
    
    // Ajouter une légende si beaucoup de chaînes
    if (d3State.visibleChannels.size > 8) {
        addLegend(g, chartWidth);
    }
    
    // Ajouter les interactions
    addD3Interactions();
}

// Ajouter une légende
function addLegend(g, chartWidth) {
    const legend = g.append('g')
        .attr('class', 'legend')
        .attr('transform', `translate(${chartWidth + 20}, 0)`);
    
    const legendItems = legend.selectAll('.legend-item')
        .data(Array.from(d3State.visibleChannels).slice(0, 15))
        .enter()
        .append('g')
        .attr('class', 'legend-item')
        .attr('transform', (d, i) => `translate(0, ${i * 20})`)
        .style('cursor', 'pointer')
        .on('click', function(event, d) {
            toggleChannelVisibility(d);
        });
    
    legendItems.append('rect')
        .attr('width', 12)
        .attr('height', 12)
        .attr('fill', d => CHANNEL_COLORS[d] || '#000')
        .attr('rx', 2);
    
    legendItems.append('text')
        .attr('x', 18)
        .attr('y', 10)
        .text(d => d)
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '10px')
        .style('fill', '#333');
}

// Basculer la visibilité d'une chaîne
function toggleChannelVisibility(channel) {
    if (d3State.visibleChannels.has(channel)) {
        d3State.visibleChannels.delete(channel);
    } else {
        d3State.visibleChannels.add(channel);
    }
    updateD3Chart();
}

// Initialiser les contrôles D3
function initD3Controls() {
    const container = document.getElementById('d3ChartContainer');
    const controlsContainer = document.createElement('div');
    controlsContainer.className = 'd3-controls';
    controlsContainer.innerHTML = `
        <div class="year-control">
            <label for="yearSlider">
                <span class="label-text">Année: </span>
                <span id="currentYearDisplay" class="year-value">${d3State.currentYear}</span>
            </label>
            <div class="slider-container">
                <input type="range" id="yearSlider" min="1995" max="2019" value="${d3State.currentYear}" step="1">
                <div class="slider-ticks">
                    <span>1995</span>
                    <span>2000</span>
                    <span>2005</span>
                    <span>2010</span>
                    <span>2015</span>
                    <span>2019</span>
                </div>
            </div>
            <div class="control-buttons">
                <button id="playAnimation" class="tv-button animation-btn">
                    <span class="button-icon">▶</span>
                    <span class="button-text">Animation</span>
                </button>
                <button id="resetView" class="tv-button reset-btn">
                    <span class="button-icon">↺</span>
                    <span class="button-text">Réinitialiser</span>
                </button>
            </div>
        </div>
        <div class="channel-filter">
            <h4 class="filter-title">Filtrer les chaînes:</h4>
            <div class="filter-buttons">
                <button class="filter-btn active" data-filter="all">Toutes</button>
                <button class="filter-btn" data-filter="public">Publiques</button>
                <button class="filter-btn" data-filter="private">Privées</button>
                <button class="filter-btn" data-filter="info">Info</button>
                <button class="filter-btn" data-filter="top5">Top 5</button>
            </div>
            <div class="visibility-controls">
                <button id="showAll" class="small-btn">Tout afficher</button>
                <button id="hideAll" class="small-btn">Tout masquer</button>
            </div>
        </div>
    `;
    
    container.appendChild(controlsContainer);
    
    // Événement slider
    const yearSlider = document.getElementById('yearSlider');
    const yearDisplay = document.getElementById('currentYearDisplay');
    
    if (yearSlider && yearDisplay) {
        yearSlider.addEventListener('input', function() {
            d3State.currentYear = parseInt(this.value);
            yearDisplay.textContent = d3State.currentYear;
            updateD3Chart();
        });
    }
    
    // Événement bouton animation
    const playButton = document.getElementById('playAnimation');
    if (playButton) {
        playButton.addEventListener('click', function() {
            if (!d3State.isAnimating) {
                animateThroughYears();
            }
        });
    }
    
    // Événement réinitialisation
    const resetButton = document.getElementById('resetView');
    if (resetButton) {
        resetButton.addEventListener('click', function() {
            d3State.currentYear = 1995;
            d3State.visibleChannels = new Set(ALL_CHANNELS);
            if (yearSlider && yearDisplay) {
                yearSlider.value = 1995;
                yearDisplay.textContent = 1995;
            }
            updateD3Chart();
        });
    }
    
    // Événements filtres de chaînes
    document.querySelectorAll('.filter-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const filter = this.getAttribute('data-filter');
            filterChannels(filter);
            
            // Mettre à jour les boutons actifs
            document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
        });
    });
    
    // Événements contrôles de visibilité
    const showAllButton = document.getElementById('showAll');
    const hideAllButton = document.getElementById('hideAll');
    
    if (showAllButton) {
        showAllButton.addEventListener('click', () => {
            d3State.visibleChannels = new Set(ALL_CHANNELS);
            updateD3Chart();
        });
    }
    
    if (hideAllButton) {
        hideAllButton.addEventListener('click', () => {
            d3State.visibleChannels.clear();
            updateD3Chart();
        });
    }
}

// Filtrer les chaînes par catégorie
function filterChannels(category) {
    d3State.visibleChannels.clear();
    
    if (category === 'all') {
        ALL_CHANNELS.forEach(channel => d3State.visibleChannels.add(channel));
    } else if (category === 'public') {
        const publicChannels = ALL_CHANNELS.filter(channel => 
            channel.includes('FRANCE') || 
            channel === 'ARTE' || 
            channel === 'France Ô'
        );
        publicChannels.forEach(channel => d3State.visibleChannels.add(channel));
    } else if (category === 'private') {
        const privateChannels = ALL_CHANNELS.filter(channel => 
            !channel.includes('FRANCE') && 
            channel !== 'ARTE' && 
            channel !== 'France Ô' &&
            channel !== 'France 24'
        );
        privateChannels.forEach(channel => d3State.visibleChannels.add(channel));
    } else if (category === 'info') {
        const infoChannels = ['BFMTV', 'CNEWS', 'LCI', 'franceinfo', 'France 24'];
        infoChannels.forEach(channel => d3State.visibleChannels.add(channel));
    } else if (category === 'top5') {
        const topChannels = ['FRANCE 2', 'FRANCE 3', 'TF1', 'M6', 'CANAL PLUS'];
        topChannels.forEach(channel => d3State.visibleChannels.add(channel));
    }
    
    updateD3Chart();
}

// Animer à travers les années
function animateThroughYears() {
    if (d3State.isAnimating) return;
    
    d3State.isAnimating = true;
    const playButton = document.getElementById('playAnimation');
    if (playButton) {
        playButton.innerHTML = '<span class="button-icon blinking">⏸</span><span class="button-text">En cours...</span>';
        playButton.classList.add('active');
    }
    
    const yearSlider = document.getElementById('yearSlider');
    const yearDisplay = document.getElementById('currentYearDisplay');
    
    let year = 1995;
    const interval = setInterval(() => {
        d3State.currentYear = year;
        if (yearSlider) yearSlider.value = year;
        if (yearDisplay) yearDisplay.textContent = year;
        
        updateD3Chart();
        
        if (year >= 2019) {
            clearInterval(interval);
            d3State.isAnimating = false;
            if (playButton) {
                playButton.innerHTML = '<span class="button-icon">▶</span><span class="button-text">Animation</span>';
                playButton.classList.remove('active');
            }
        }
        
        year++;
    }, 100);
}

// Animer jusqu'à une année spécifique
function animateToYear(targetYear, duration = D3_CONFIG.animationDuration) {
    const startYear = d3State.currentYear;
    const startTime = Date.now();
    
    function animate() {
        const elapsed = Date.now() - startTime;
        const progress = Math.min(elapsed / duration, 1);
        
        const easeProgress = 1 - Math.pow(1 - progress, 3);
        d3State.currentYear = Math.floor(startYear + (targetYear - startYear) * easeProgress);
        
        const yearSlider = document.getElementById('yearSlider');
        const yearDisplay = document.getElementById('currentYearDisplay');
        
        if (yearSlider && yearDisplay) {
            yearSlider.value = d3State.currentYear;
            yearDisplay.textContent = d3State.currentYear;
        }
        
        updateD3Chart();
        
        if (progress < 1) {
            requestAnimationFrame(animate);
        }
    }
    
    requestAnimationFrame(animate);
}

// Ajouter des interactions D3
function addD3Interactions() {
    const svg = d3.select('#d3ChartContainer svg.tv-d3-chart');
    
    // Créer un tooltip
    const tooltip = d3.select('#d3ChartContainer')
        .append('div')
        .attr('class', 'd3-tooltip')
        .style('opacity', 0)
        .style('position', 'absolute')
        .style('background', 'white')
        .style('border', '1px solid #0066cc')
        .style('border-radius', '4px')
        .style('padding', '12px')
        .style('pointer-events', 'none')
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '13px')
        .style('box-shadow', '0 4px 12px rgba(0,0,0,0.15)')
        .style('z-index', '1000')
        .style('min-width', '200px');
    
    // Interactions avec les lignes
    svg.selectAll('.tv-line')
        .on('mouseover', function(event, d) {
            d3.select(this)
                .attr('stroke-width', 4)
                .attr('opacity', 1);
            
            if (d.data.length > 0) {
                const values = d.data.map(point => point.value);
                const currentValue = d.data[d.data.length - 1].value;
                const startValue = d.data[0].value;
                const evolution = currentValue - startValue;
                const avgValue = d3.mean(values) || 0;
                
                tooltip.transition()
                    .duration(200)
                    .style('opacity', 0.98);
                
                tooltip.html(`
                    <div style="margin-bottom: 8px;">
                        <strong style="color: ${d.color}; font-size: 14px;">${d.channel}</strong>
                    </div>
                    <div style="border-bottom: 1px solid #eee; padding-bottom: 8px; margin-bottom: 8px;">
                        <div><strong>${d3State.currentYear}:</strong> ${currentValue.toFixed(1)}%</div>
                        <div><strong>1995:</strong> ${startValue.toFixed(1)}%</div>
                    </div>
                    <div>
                        <div><strong>Évolution:</strong> <span style="color: ${evolution >= 0 ? '#2ecc71' : '#e74c3c'}">${evolution >= 0 ? '+' : ''}${evolution.toFixed(1)} points</span></div>
                        <div><strong>Moyenne:</strong> ${avgValue.toFixed(1)}%</div>
                        <div><strong>Temps hommes:</strong> ${(100 - currentValue).toFixed(1)}%</div>
                    </div>
                `)
                .style('left', (event.pageX + 15) + 'px')
                .style('top', (event.pageY - 10) + 'px');
            }
        })
        .on('mouseout', function(event, d) {
            d3.select(this)
                .attr('stroke-width', 2)
                .attr('opacity', 0.7);
            
            tooltip.transition()
                .duration(200)
                .style('opacity', 0);
        })
        .on('click', function(event, d) {
            toggleChannelVisibility(d.channel);
        });
    
    // Interactions avec les points
    svg.selectAll('.data-point')
        .on('mouseover', function(event, d) {
            d3.select(this)
                .attr('r', 5)
                .attr('opacity', 1);
            
            tooltip.transition()
                .duration(200)
                .style('opacity', 0.98);
            
            tooltip.html(`
                <div style="margin-bottom: 8px;">
                    <strong style="color: ${d.color}; font-size: 14px;">${d.channel}</strong>
                </div>
                <div style="border-bottom: 1px solid #eee; padding-bottom: 8px; margin-bottom: 8px;">
                    <div><strong>Année:</strong> ${d.year}</div>
                </div>
                <div>
                    <div><strong>Temps de parole femmes:</strong> ${d.value.toFixed(1)}%</div>
                    <div><strong>Temps de parole hommes:</strong> ${(100 - d.value).toFixed(1)}%</div>
                    <div><strong>Ratio F/H:</strong> ${(d.value / Math.max(1, 100 - d.value)).toFixed(2)}</div>
                </div>
            `)
            .style('left', (event.pageX + 15) + 'px')
            .style('top', (event.pageY - 10) + 'px');
        })
        .on('mouseout', function(event, d) {
            d3.select(this)
                .attr('r', 3)
                .attr('opacity', 0.8);
            
            tooltip.transition()
                .duration(200)
                .style('opacity', 0);
        });
}

// Ajouter le CSS pour D3
function addD3Styles() {
    const style = document.createElement('style');
    style.textContent = `
        #d3ChartContainer {
            font-family: 'Segoe UI', sans-serif;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 12px;
            margin: 20px 0;
        }
        
        .d3-chart-wrapper {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .chart-header h3 {
            margin: 0 0 20px 0;
            color: #333;
            font-size: 18px;
            text-align: center;
        }
        
        .chart-svg-container {
            overflow: auto;
            margin-bottom: 20px;
        }
        
        .tv-d3-chart {
            background: white;
            border-radius: 8px;
            border: 1px solid #e0e0e0;
        }
        
        .tv-d3-chart .axis path,
        .tv-d3-chart .axis line {
            stroke: #ccc;
            stroke-width: 1;
            shape-rendering: crispEdges;
        }
        
        .tv-d3-chart .axis text {
            fill: #666;
            font-family: 'Segoe UI', sans-serif;
            font-size: 12px;
        }
        
        .tv-d3-chart .tv-line {
            transition: stroke-width 0.3s ease;
        }
        
        .tv-d3-chart .tv-line:hover {
            stroke-width: 4 !important;
            opacity: 1 !important;
            cursor: pointer;
        }
        
        .d3-controls {
            margin-top: 20px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e0e0e0;
        }
        
        .year-control {
            display: flex;
            flex-direction: column;
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .year-control label {
            font-size: 14px;
            color: #333;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .year-value {
            font-size: 16px;
            color: #0066cc;
            font-weight: 700;
            min-width: 40px;
        }
        
        .slider-container {
            position: relative;
            padding: 10px 0;
        }
        
        #yearSlider {
            width: 100%;
            height: 6px;
            border-radius: 3px;
            background: #ddd;
            outline: none;
            -webkit-appearance: none;
        }
        
        #yearSlider::-webkit-slider-thumb {
            -webkit-appearance: none;
            width: 20px;
            height: 20px;
            border-radius: 50%;
            background: #0066cc;
            cursor: pointer;
            border: 3px solid white;
            box-shadow: 0 2px 6px rgba(0,0,0,0.2);
        }
        
        .slider-ticks {
            display: flex;
            justify-content: space-between;
            margin-top: 5px;
            padding: 0 5px;
        }
        
        .slider-ticks span {
            font-size: 11px;
            color: #666;
        }
        
        .control-buttons {
            display: flex;
            gap: 10px;
        }
        
        .tv-button {
            padding: 8px 16px;
            background: #0066cc;
            border: none;
            border-radius: 4px;
            color: white;
            font-family: 'Segoe UI', sans-serif;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: background 0.3s ease;
        }
        
        .tv-button:hover {
            background: #0052a3;
        }
        
        .tv-button.active {
            background: #00a86b;
        }
        
        .reset-btn {
            background: #666;
        }
        
        .reset-btn:hover {
            background: #555;
        }
        
        .button-icon {
            font-size: 12px;
        }
        
        .blinking {
            animation: blink 1s infinite;
        }
        
        @keyframes blink {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .channel-filter {
            padding-top: 15px;
            border-top: 1px solid #e0e0e0;
        }
        
        .filter-title {
            font-size: 14px;
            color: #333;
            margin-bottom: 12px;
            font-weight: 600;
        }
        
        .filter-buttons {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
            margin-bottom: 15px;
        }
        
        .filter-btn {
            padding: 6px 12px;
            background: #f0f0f0;
            border: 1px solid #ddd;
            border-radius: 4px;
            color: #666;
            font-family: 'Segoe UI', sans-serif;
            font-size: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .filter-btn:hover {
            background: #e0e0e0;
        }
        
        .filter-btn.active {
            background: #0066cc;
            color: white;
            border-color: #0066cc;
        }
        
        .visibility-controls {
            display: flex;
            gap: 8px;
        }
        
        .small-btn {
            padding: 4px 12px;
            background: #f8f9fa;
            border: 1px solid #ddd;
            border-radius: 4px;
            color: #666;
            font-family: 'Segoe UI', sans-serif;
            font-size: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .small-btn:hover {
            background: #e9ecef;
        }
        
        .d3-tooltip {
            pointer-events: none;
            position: absolute;
            z-index: 1000;
            background: white;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 12px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            font-family: 'Segoe UI', sans-serif;
            font-size: 13px;
            line-height: 1.5;
            max-width: 250px;
        }
        
        .legend {
            font-family: 'Segoe UI', sans-serif;
        }
        
        .legend-item:hover text {
            font-weight: 600;
        }
        
        .data-point:hover {
            r: 6 !important;
            cursor: pointer;
        }
    `;
    
    document.head.appendChild(style);
}

// Initialiser au chargement
document.addEventListener('DOMContentLoaded', function() {
    console.log('D3.js TV Chart: Initialisation...');
    
    // Créer le conteneur si nécessaire
    let container = document.getElementById('d3ChartContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'd3ChartContainer';
        document.body.appendChild(container);
    }
    
    // Ajouter les styles
    addD3Styles();
    
    // Attendre que D3 soit chargé
    const checkD3 = setInterval(() => {
        if (typeof d3 !== 'undefined') {
            clearInterval(checkD3);
            console.log('D3.js chargé, initialisation du graphique...');
            setTimeout(() => {
                initD3Chart();
            }, 100);
        }
    }, 100);
    
    // Timeout de sécurité
    setTimeout(() => {
        if (typeof d3 === 'undefined') {
            console.error('D3.js n\'a pas pu être chargé');
            const errorDiv = document.createElement('div');
            errorDiv.innerHTML = `
                <div style="padding: 20px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 4px; margin: 20px;">
                    <h3 style="color: #856404; margin-top: 0;">Erreur de chargement</h3>
                    <p style="color: #856404;">D3.js n'est pas chargé. Veuillez inclure la bibliothèque D3.js avant ce script.</p>
                    <p style="color: #856404; font-size: 14px;">Ajoutez cette ligne dans votre HTML :</p>
                    <code style="background: #f8f9fa; padding: 10px; border-radius: 4px; display: block; margin-top: 10px;">
                        &lt;script src="https://d3js.org/d3.v7.min.js"&gt;&lt;/script&gt;
                    </code>
                </div>
            `;
            container.appendChild(errorDiv);
        }
    }, 3000);
});

// Exporter les fonctions
if (typeof window !== 'undefined') {
    window.initD3Chart = initD3Chart;
    window.destroyD3Chart = function() {
        const container = document.getElementById('d3ChartContainer');
        if (container) container.innerHTML = '';
    };
    window.animateToYear = animateToYear;
    window.updateD3Chart = updateD3Chart;
}

console.log('Module D3.js TV Chart prêt');